
/*
* Copyright (C) 2016 Texas Instruments Incorporated - www.ti.com
*
*
*  Redistribution and use in source and binary forms, with or without
*  modification, are permitted provided that the following conditions
*  are met:
*
*    Redistributions of source code must retain the above copyright
*    notice, this list of conditions and the following disclaimer.
*
*    Redistributions in binary form must reproduce the above copyright
*    notice, this list of conditions and the following disclaimer in the
*    documentation and/or other materials provided with the
*    distribution.
*
*    Neither the name of Texas Instruments Incorporated nor the names of
*    its contributors may be used to endorse or promote products derived
*    from this software without specific prior written permission.
*
*  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
*  "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
*  LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
*  A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
*  OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
*  SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
*  LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
*  DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
*  THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
*  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
*  OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
*/

/* HALCoGen Register Definition Includes Files */
#include "sys_common.h"
#include "sys_pmu.h"

/* HALCoGen Drivers Include Files */
#include "crc.h"

/* CG Tools Include Files */
#include <crc_tbl.h>
#include <stdio.h>

/* Application Include Files */
#include "CRC64_calc.h"


extern const CRC_TABLE _my_crc_table;
volatile uint32 cycles = 0ul;

#ifndef crcREG
#define crcREG crcREG1
/* Note: This is just a helper to identify RM57 and LC43 devices, as they need different handling for the DMA */
#define RM57_OR_LC43
#else
#undef crcREG1
#undef crcREG2
#undef RM57_OR_LC43
#endif

volatile uint64 u64Signature  = 0ull;

#pragma WEAK(crcSetConfig)
void crcSetConfig(crcBASE_t *crc,crcConfig_t *param)
{
	switch (param->crc_channel)
	{
	case 0U:
		crc->CTRL2 &=0xFFFFFFFCU;
		crc->CTRL0 |=0x00000001U;
		crc->CTRL0 &=0xFFFFFFFEU;
		//crc->PCOUNT_REG1 = param->pcount;
		//crc->SCOUNT_REG1 = param->scount;
		//crc->WDTOPLD1= param->wdg_preload;
		//crc->BCTOPLD1= param->block_preload;
		crc->CTRL2 |= param->mode;
	    break;
	case 1U:
		crc->CTRL2 &=0xFFFFFCFFU;
		crc->CTRL0 |=0x00000100U;
		crc->CTRL0 &=0xFFFFFEFFU;
		//crc->PCOUNT_REG2 = param->pcount;
		//crc->SCOUNT_REG2= param->scount;
		//crc->WDTOPLD2= param->wdg_preload;
		//crc->BCTOPLD2= param->block_preload;
		crc->CTRL2 |= (uint32)((uint32)param->mode << 8U);
        break;
	default :
		break;
    }
}

int main(void)
{
	uint32          i;

	_pmuInit_();
	_pmuEnableCountersGlobal_();

	crcInit();

	if(1)
	{
		printf("Start CRC Calculation in SW Mode\n");

		_pmuResetCycleCounter_();
		_pmuStartCounters_(pmuCYCLE_COUNTER);

		for (i = 0ul ; i < _my_crc_table.num_recs ; i++)
		{
			//uint64   u64Signature = 0ull;
			uint64 * pu64Address  = (void *)(-1); /* NULL is a valid address */
			uint32   u32Size      = 0ul;

			/* Check Alignment of Start Address */
			if (0ull == (_my_crc_table.recs[i].addr % 8ull))
			{
				pu64Address = (uint64*)(_my_crc_table.recs[i].addr);

				/* Check Alignment of Size */
				if (0ul == (_my_crc_table.recs[i].size % 8ul))
				{
					/* Assign and adjust size to increments of 8 Bytes */
					u32Size = _my_crc_table.recs[i].size / 8ul;

					/* Compute CRC in SW */
					u64Signature = calc_crc64(pu64Address, u32Size);

					/* Check CRC */
					if (u64Signature != _my_crc_table.recs[i].crc_value)
					{
						/* Fail */

						printf("Wrong CRC in SW Module @ address: 0x%08X\n", _my_crc_table.recs[i].addr);
						while(1);
					}
					else
					{
						/* Pass */
					}
				}
				else
				{
					/* Wrong Alignment of Size */
				}
			}
			else
			{
				/* Wrong Alignment of Start Address */
			}
		}

		cycles = _pmuGetCycleCount_();
		_pmuStopCounters_(pmuCYCLE_COUNTER);
		printf("Finished CRC calculation, CPU cycles: %d\n", cycles);
	}

	if(1)
	{
		crcConfig_t     sCrcParams;

		printf("Start CRC Calculation in Full CPU Mode\n");

		sCrcParams.crc_channel   = CRC_CH1;
		sCrcParams.mode          = CRC_FULL_CPU;
		sCrcParams.pcount        = 0u; /* All counters are disabled in Full CPU mode */
		sCrcParams.scount        = 0u; /* All counters are disabled in Full CPU mode */
		sCrcParams.wdg_preload   = 0u; /* All counters are disabled in Full CPU mode */
		sCrcParams.block_preload = 0u; /* All counters are disabled in Full CPU mode */

		crcSetConfig(crcREG, &sCrcParams);

		_pmuResetCycleCounter_();
		_pmuStartCounters_(pmuCYCLE_COUNTER);

		for (i = 0ul ; i < _my_crc_table.num_recs ; i++)
		{
			crcModConfig_t  sCrcModConfig;
			//volatile uint64 u64Signature  = 0ull;

			sCrcModConfig.crc_channel  = CRC_CH1;
			sCrcModConfig.mode         = CRC_FULL_CPU;
			sCrcModConfig.data_length  = (_my_crc_table.recs[i].size + 7ul) / 8ul;
			sCrcModConfig.src_data_pat = (uint64*)_my_crc_table.recs[i].addr;

			/* Compute CRC with HW Module, Full CPU Mode */
			crcChannelReset(crcREG, CRC_CH1);
			crcSignGen(crcREG, &sCrcModConfig);
			u64Signature = crcGetPSASig(crcREG, CRC_CH1);

			/* Check CRC */
			if (u64Signature != _my_crc_table.recs[i].crc_value)
			{
				printf("Wrong CRC in SW Module @ address: 0x%08X\n", _my_crc_table.recs[i].addr);
				while(1); /* Fail */
			}
			else
			{
				/* Pass */
			}
		}

		cycles = _pmuGetCycleCount_();
		_pmuStopCounters_(pmuCYCLE_COUNTER);
		printf("Finished CRC calculation, CPU cycles: %d\n", cycles);
	}

	while(1); /* Loop for ever. */
	
	return 0;
}
